<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class LegallyMail_API {

    private $api_token;
    private $api_base_url;

    public function __construct() {
        $options = get_option('legallymail_options', array());
        $this->api_token = isset($options['api_token']) ? $options['api_token'] : '';
        $this->api_base_url = 'https://legallymail.com'; // Change this if needed based on constant
    }

    public function send_certified_email( $to, $subject, $message, $attachments = array() ) {
    
        if ( empty( $this->api_token ) ) {
            return new WP_Error( 'not_configured', 'API Token is missing.' );
        }

        $url = $this->api_base_url . '/api/send';

        $body = array(
            'token' => $this->api_token,
            'recipient' => $to,
            'subject' => $subject,
            'message' => $message,
            'attachments' => array()
        );

        if ( ! empty( $attachments ) ) {
            foreach ( $attachments as $file_path ) {
                if ( file_exists( $file_path ) ) {
                    $file_content = file_get_contents( $file_path );
                    $body['attachments'][] = array(
                        'name' => basename( $file_path ),
                        'content' => base64_encode( $file_content ),
                        'type' => mime_content_type( $file_path )
                    );
                }
            }
        }

        $args = array(
            'body'    => json_encode( $body ),
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_token,
                'Content-Type'  => 'application/json',
            ),
            'timeout' => 45,
            'method' => 'POST'
        );

        $response = wp_remote_post( $url, $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );

        if ( isset( $data['success'] ) && $data['success'] ) {
            return $data;
        } else {
            $error_msg = isset( $data['error'] ) ? $data['error'] : 'Unknown API error. Response: ' . $response_body;
            return new WP_Error( 'api_error', $error_msg );
        }
    }

    public function get_email_status( $tracking_id ) {
        if ( empty( $this->api_token ) ) {
            return new WP_Error( 'not_configured', 'API Token is missing.' );
        }

        $url = $this->api_base_url . '/api/email';
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_token,
                'Content-Type'  => 'application/json',
            ),
            'timeout' => 45,
            'method' => 'GET'
        );
        
        // Append query args properly
        $url = add_query_arg( 'tracking_id', $tracking_id, $url );

        $response = wp_remote_get( $url, $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );

        if ( isset( $data['success'] ) && $data['success'] ) {
            return $data['email'];
        } else {
            $error_msg = isset( $data['error'] ) ? $data['error'] : 'Unknown API error';
            return new WP_Error( 'api_error', $error_msg );
        }
    }

    public function get_user_info() {
        if ( empty( $this->api_token ) ) {
            return new WP_Error( 'not_configured', 'API Token is missing.' );
        }

        $url = $this->api_base_url . '/api/info';
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_token,
                'Content-Type'  => 'application/json',
            ),
            'timeout' => 45,
            'method' => 'GET'
        );

        $response = wp_remote_get( $url, $args );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $response_body = wp_remote_retrieve_body( $response );
        $data = json_decode( $response_body, true );

        if ( isset( $data['success'] ) && $data['success'] === false ) {
            $error_msg = isset( $data['error'] ) ? $data['error'] : 'Unknown API error';
            return new WP_Error( 'api_error', $error_msg );
        }

        return $data;
    }
}
