<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class LegallyMail_Admin {

    private $api;

    public function __construct() {
        $this->api = new LegallyMail_API();
    }

    public function run() {
        add_action('admin_menu', array($this, 'add_plugin_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        
        // Add column to users list
        add_filter('manage_users_columns', array($this, 'add_user_columns'));
        add_filter('manage_users_custom_column', array($this, 'show_user_column_content'), 10, 3);

        // AJAX for bulk status update
        add_action('wp_ajax_legallymail_update_status', array($this, 'ajax_update_status'));

        // Check for DB updates
        add_action('admin_init', array($this, 'update_db_check'));

        // Bulk Users Action
        add_filter('bulk_actions-users', array($this, 'register_bulk_user_actions'));
        add_filter('handle_bulk_actions-users', array($this, 'handle_bulk_user_actions'), 10, 3);

        // WooCommerce Integration
        add_filter('bulk_actions-edit-shop_order', array($this, 'register_bulk_order_actions'));
        add_filter('handle_bulk_actions-edit-shop_order', array($this, 'handle_bulk_order_actions'), 10, 3);
        
        // AJAX for Templates
        add_action('wp_ajax_legallymail_save_template', array($this, 'ajax_save_template'));
        add_action('wp_ajax_legallymail_delete_template', array($this, 'ajax_delete_template'));

        // WooCommerce Single Order Action
        add_action('add_meta_boxes', array($this, 'add_order_meta_box'));
    }

    public function update_db_check() {
        $installed_ver = get_option('legallymail_db_version');
        if ($installed_ver !== LEGALLYMAIL_VERSION) {
            self::create_db_tables();
            update_option('legallymail_db_version', LEGALLYMAIL_VERSION);
        }
    }

    /**
     * Create the database table to track emails
     */
    public static function create_db_tables() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'legallymail_emails';
        
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            time datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
            recipient varchar(255) NOT NULL,
            subject varchar(255) NOT NULL,
            tracking_id varchar(100) NOT NULL,
            status varchar(50) DEFAULT 'queued' NOT NULL,
            api_response text,
            error_status text,
            PRIMARY KEY  (id)
        ) $charset_collate;";


        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql );

        // Create Templates Table
        $table_templates = $wpdb->prefix . 'legallymail_templates';
        $sql_templates = "CREATE TABLE $table_templates (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            subject varchar(255) NOT NULL,
            message longtext NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";
        dbDelta( $sql_templates );
    }

    public function add_plugin_admin_menu() {
        add_menu_page(
            'LegallyMail Certified Email', 
            'LegallyMail', 
            'manage_options', 
            'legallymail-certified-email', 
            array($this, 'display_email_list_page'), 
            'dashicons-email-alt', 
            26
        );

        // Rename the first submenu item (which defaults to the parent name)
        add_submenu_page(
            'legallymail-certified-email',
            'Email List',
            'Email List',
            'manage_options',
            'legallymail-certified-email',
            array($this, 'display_email_list_page')
        );

        add_submenu_page(
            'legallymail-certified-email', 
            'Send New Email', 
            'Send New', 
            'manage_options', 
            'legallymail-send-email', 
            array($this, 'display_send_email_page')
        );

        add_submenu_page(
            'legallymail-certified-email', 
            'Settings', 
            'Settings', 
            'manage_options', 
            'legallymail-settings', 
            array($this, 'display_plugin_setup_page')
        );
    }

    public function register_settings() {
        register_setting( 'legallymail_options_group', 'legallymail_options', array( $this, 'sanitize_settings' ) );

        add_settings_section(
            'legallymail_api_section', 
            'API Configuration', 
            null, 
            'legallymail-settings'
        );

        add_settings_field(
            'api_token', 
            'API Token', 
            array( $this, 'api_token_callback' ), 
            'legallymail-settings', 
            'legallymail_api_section'
        );
    }

    public function sanitize_settings( $input ) {
        $new_input = array();
        if( isset( $input['api_token'] ) )
            $new_input['api_token'] = sanitize_text_field( $input['api_token'] );

        return $new_input;
    }

    public function api_token_callback() {
        $options = get_option( 'legallymail_options' );
        $token = isset( $options['api_token'] ) ? esc_attr( $options['api_token'] ) : '';
        echo '<input type="text" name="legallymail_options[api_token]" value="' . esc_attr( $token ) . '" class="regular-text" />';
        echo '<p class="description">Enter your API Token from the LegallyMail dashboard. <a href="https://legallymail.com/api-tokens" target="_blank">Get your API Token here</a>.</p>';
    }

    public function display_plugin_setup_page() {
        ?>
        <div class="wrap">
            <h1>LegallyMail Settings</h1>
            <form action="options.php" method="post">
                <?php 
                settings_fields( 'legallymail_options_group' );
                do_settings_sections( 'legallymail-settings' );
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    public function add_user_columns($columns) {
        $columns['legallymail_action'] = 'Certified Email';
        return $columns;
    }

    public function show_user_column_content($value, $column_name, $user_id) {
        if ( 'legallymail_action' == $column_name ) {
            $user_info = get_userdata($user_id);
            $email = $user_info->user_email;
            $send_url = admin_url('admin.php?page=legallymail-send-email&recipient_email=' . urlencode($email));
            return '<a href="' . esc_url($send_url) . '" class="button button-secondary"><span class="dashicons dashicons-email-alt" style="margin-top: 3px; font-size: 16px;"></span> ' . esc_html__( 'Send Certified Email', 'legallymail-certified-email' ) . '</a>';
        }
        return $value;
        return $value;
    }

    public function register_bulk_user_actions($actions) {
        $actions['legallymail_bulk_send'] = __('Send Certified Email', 'legallymail-certified-email');
        return $actions;
    }

    public function handle_bulk_user_actions($redirect_to, $doaction, $user_ids) {
        if ($doaction !== 'legallymail_bulk_send') {
            return $redirect_to;
        }

        $emails = array();
        foreach ($user_ids as $user_id) {
            $user = get_userdata($user_id);
            if ($user && !empty($user->user_email)) {
                $emails[] = $user->user_email;
            }
        }

        if (!empty($emails)) {
            $redirect_to = admin_url('admin.php?page=legallymail-send-email&recipient_email=' . urlencode(implode(',', $emails)));
        }

        return $redirect_to;
    }

    public function register_bulk_order_actions($actions) {
        $actions['legallymail_bulk_order_send'] = __('Send Certified Email', 'legallymail-certified-email');
        return $actions;
    }

    public function handle_bulk_order_actions($redirect_to, $doaction, $post_ids) {
        if ($doaction !== 'legallymail_bulk_order_send') {
            return $redirect_to;
        }

        $emails = array();
        foreach ($post_ids as $post_id) {
            $order = wc_get_order($post_id);
            if ($order) {
                $email = $order->get_billing_email();
                if (!empty($email)) {
                    $emails[] = $email;
                }
            }
        }

        if (!empty($emails)) {
            $redirect_to = admin_url('admin.php?page=legallymail-send-email&recipient_email=' . urlencode(implode(',', array_unique($emails))));
        }

        return $redirect_to;
    }

    public function add_order_meta_box() {
        add_meta_box(
            'legallymail_order_action',
            'LegallyMail Certified Email',
            array($this, 'render_order_meta_box'),
            'shop_order',
            'side',
            'default'
        );
    }

    public function render_order_meta_box($post) {
        $order = wc_get_order($post->ID);
        $email = $order->get_billing_email();
        if ($email) {
            $send_url = admin_url('admin.php?page=legallymail-send-email&recipient_email=' . urlencode($email));
            echo '<a href="' . esc_url($send_url) . '" class="button button-primary" style="width:100%;text-align:center;">Send Certified Email</a>';
            echo '<p class="description" style="margin-top:5px;">Recipient: ' . esc_html($email) . '</p>';
        } else {
            echo '<p>No billing email found.</p>';
        }
    }

    public function ajax_save_template() {
        check_ajax_referer('legallymail_template_action', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'legallymail_templates';
        
        $name = sanitize_text_field($_POST['name']);
        $subject = sanitize_text_field($_POST['subject']);
        $message = wp_kses_post($_POST['message']);

        if (empty($name) || empty($subject) || empty($message)) {
            wp_send_json_error('All fields are required');
        }

        $wpdb->insert(
            $table_name,
            array(
                'name' => $name,
                'subject' => $subject,
                'message' => $message
            )
        );

        wp_send_json_success(array('id' => $wpdb->insert_id, 'name' => $name));
    }

    public function ajax_delete_template() {
        check_ajax_referer('legallymail_template_action', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'legallymail_templates';
        $id = intval($_POST['id']);

        $wpdb->delete($table_name, array('id' => $id));
        wp_send_json_success();
    }

    public function display_send_email_page() {
        $user_info = $this->api->get_user_info();
        $available_credits = 'Not available';
        if (!is_wp_error($user_info) && isset($user_info['user']['remaining_emails'])) {
            $available_credits = $user_info['user']['remaining_emails'];
        }

        // Handle Form Submission
        if ( isset( $_POST['legallymail_send_nonce'] ) && wp_verify_nonce( sanitize_key( wp_unslash( $_POST['legallymail_send_nonce'] ) ), 'legallymail_send_action' ) ) {
            $to      = isset( $_POST['recipient_email'] ) ? sanitize_text_field( wp_unslash( $_POST['recipient_email'] ) ) : '';
            $subject = isset( $_POST['subject'] ) ? sanitize_text_field( wp_unslash( $_POST['subject'] ) ) : '';
            $message = isset( $_POST['message'] ) ? wp_kses_post( wp_unslash( $_POST['message'] ) ) : '';

            // Process attachments
            $attachments = array();
            $uploaded_files = array(); // Keep track for cleanup
            $upload_error = false;

            if ( ! empty( $_FILES['legallymail_attachments']['name'][0] ) ) {
                $files = $_FILES['legallymail_attachments'];
                $count = count( $files['name'] );
                
                // Validate Count (Max 4)
                if ( $count > 4 ) {
                    echo '<div class="notice notice-error"><p>Error: Maximum 4 files allowed.</p></div>';
                    $upload_error = true;
                }

                // Validate Total Size (Max 15MB)
                $total_size = 0;
                foreach ( $files['size'] as $size ) {
                    $total_size += $size;
                }
                if ( $total_size > 15 * 1024 * 1024 ) {
                    echo '<div class="notice notice-error"><p>Error: Total file size exceeds 15 MB.</p></div>';
                    $upload_error = true;
                }

                if ( ! $upload_error ) {
                    // Use WordPress upload directory for temp storage to ensure write permissions
                    $upload_dir = wp_upload_dir();
                    $temp_dir   = $upload_dir['basedir'] . '/legallymail_temp/';
                    
                    global $wp_filesystem;
                    if ( empty( $wp_filesystem ) ) {
                        require_once ABSPATH . 'wp-admin/includes/file.php';
                        WP_Filesystem();
                    }

                    if ( ! file_exists( $temp_dir ) ) {
                        $wp_filesystem->mkdir( $temp_dir );
                    }

                    for ( $i = 0; $i < $count; $i++ ) {
                        if ( $files['error'][$i] === 0 ) {
                            $original_name = sanitize_file_name( $files['name'][$i] );
                            $temp_path     = $temp_dir . uniqid() . '_' . $original_name;
                            
                            if ( $wp_filesystem->move( $files['tmp_name'][$i], $temp_path ) ) {
                                $attachments[]    = $temp_path;
                                $uploaded_files[] = $temp_path;
                            } else {
                                /* translators: %s: file name */
                                echo '<div class="notice notice-error"><p>' . sprintf( esc_html__( 'Error uploading file: %s', 'legallymail-certified-email' ), esc_html( $files['name'][$i] ) ) . '</p></div>';
                                $upload_error = true;
                                break;
                            }
                        }
                    }
                }
            }

            if ( ! $upload_error ) {
                $result = $this->api->send_certified_email( $to, $subject, $message, $attachments );

                if ( is_wp_error( $result ) ) {
                    echo '<div class="notice notice-error"><p>Error: ' . esc_html( $result->get_error_message() ) . '</p></div>';
                } else {
                    // Extract Tracking ID (handles both test mode and production structure)
                    $tracking_id = '';
                    if ( isset( $result['data']['tracking_id'] ) ) {
                        $tracking_id = $result['data']['tracking_id'];
                    } elseif ( isset( $result['tracking_id'] ) ) {
                        $tracking_id = $result['tracking_id'];
                    }
    
                    if ( ! empty( $tracking_id ) ) {
                        // Save to DB
                        global $wpdb;
                        $table_name = $wpdb->prefix . 'legallymail_emails';
                        
                        $recipient_list = array_map('trim', explode(',', $to));
                        $success_inserted = true;
                        
                        foreach ($recipient_list as $single_recipient) {
                            if (empty($single_recipient)) continue;
                            
                            $local_status = 'sent';
                            $local_error = null;

                            // Try to find specific status for this recipient in the API result
                            if (isset($result['data']['recipients']) && is_array($result['data']['recipients'])) {
                                foreach ($result['data']['recipients'] as $res_recipient) {
                                    // Match by email (cleaning it just in case)
                                    $clean_res_email = strtolower(trim($res_recipient['email']));
                                    $clean_single_email = strtolower(trim($single_recipient));
                                    
                                    // If single_recipient is "Name <email@...>", extract email
                                    if (preg_match('/<(.+)>$/', $clean_single_email, $matches)) {
                                        $clean_single_email = strtolower(trim($matches[1]));
                                    }

                                    if ($clean_res_email === $clean_single_email) {
                                        $local_status = $res_recipient['status'];
                                        $local_error = isset($res_recipient['error_details']) ? $res_recipient['error_details'] : null;
                                        break;
                                    }
                                }
                            }

                            // Fallback to top-level error if local_error is still null and status is failed
                            if ($local_status === 'failed' && empty($local_error)) {
                                $local_error = isset($result['data']['error_status']) ? $result['data']['error_status'] : null;
                            }

                            $inserted = $wpdb->insert( 
                                $table_name, 
                                array( 
                                    'time' => current_time( 'mysql' ), 
                                    'recipient' => $single_recipient, 
                                    'subject' => $subject,
                                    'tracking_id' => $tracking_id,
                                    'status' => $local_status,
                                    'api_response' => json_encode($result),
                                    'error_status' => $local_error
                                ) 
                            );
                            if ($inserted === false) $success_inserted = false;
                        }
    
                        if ( $success_inserted === false ) {
                            echo '<div class="notice notice-error"><p>Error saving details to database: ' . esc_html( $wpdb->last_error ) . '</p></div>';
                        } else {
                            echo '<div class="notice notice-success"><p>Certified Email sent successfully! Redirecting...</p></div>';
                            echo '<script>window.location.href = "' . esc_url_raw( admin_url('admin.php?page=legallymail-certified-email&message=sent') ) . '";</script>';
                        }
                    } else {
                        echo '<div class="notice notice-warning"><p>Email sent but could not retrieve Tracking ID. Response: ' . esc_html(json_encode($result)) . '</p></div>';
                    }
                }
            }

            // Cleanup temp files
            foreach ( $uploaded_files as $file ) {
                if ( file_exists( $file ) ) {
                    wp_delete_file( $file );
                }
            }
        }

        ?>
        <div class="wrap">
            <h1>Send Certified Email</h1>
            
            <?php
            global $wpdb;
            $templates = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}legallymail_templates ORDER BY name ASC");
            ?>
            
            <div style="background: #fff; padding: 15px; margin: 15px 0; border: 1px solid #ccd0d4; display: flex; align-items: center; gap: 10px;">
                <label><strong>Load Template:</strong></label>
                <select id="template-selector">
                    <option value="">-- Select Template --</option>
                    <?php foreach ($templates as $tmpl): ?>
                        <option value="<?php echo esc_attr($tmpl->id); ?>" 
                                data-subject="<?php echo esc_attr($tmpl->subject); ?>" 
                                data-message="<?php echo esc_attr($tmpl->message); ?>">
                            <?php echo esc_html($tmpl->name); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <button type="button" class="button" id="btn-load-template">Load</button>
                <button type="button" class="button" id="btn-delete-template" style="color: #a00;">Delete</button>
                <span style="flex-grow: 1;"></span>
                <button type="button" class="button button-secondary" id="btn-save-template">Save Current as Template</button>
            </div>

            <div class="notice notice-info" style="display: flex; align-items: center; justify-content: space-between; padding: 10px 20px;">
                <p style="margin: 0; font-size: 16px;">
                    Available credits: <strong><?php echo esc_html($available_credits); ?></strong>
                </p>
                <a href="https://legallymail.com/subscription" target="_blank" class="button button-primary">Buy more credits</a>
            </div>

            <form method="post" action="" enctype="multipart/form-data" id="legallymail-send-form">
                <?php wp_nonce_field( 'legallymail_send_action', 'legallymail_send_nonce' ); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Recipient Email(s)</th>
                        <td>
                            <?php $recipient = isset($_GET['recipient_email']) ? sanitize_text_field( wp_unslash( $_GET['recipient_email'] ) ) : ''; ?>
                            <input type="text" name="recipient_email" id="recipient_email" value="<?php echo esc_attr($recipient); ?>" class="regular-text" placeholder="email1@example.com, email2@example.com" required />
                            <p class="description">You can enter multiple emails separated by commas. Credits needed: <strong id="credit-counter">0</strong></p>
                        </td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Subject</th>
                        <td><input type="text" name="subject" class="regular-text" required /></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Message</th>
                        <td>
                            <?php wp_editor( '', 'message', array( 'media_buttons' => false, 'textarea_rows' => 10 ) ); ?>
                        </td>
                    </tr>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Attachments</th>
                        <td>
                            <input type="file" name="legallymail_attachments[]" multiple="multiple" class="regular-text" accept=".pdf,.doc,.docx,.jpg,.jpeg,.png" />
                            <p class="description">Select up to 4 files. Total size: <strong id="file-counter">0 MB</strong> / 15 MB limit.</p>
                            <p class="description">Allowed formats: <strong>PDF, DOC, DOCX, JPG, PNG</strong>.</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Send Certified Email'); ?>
            </form>
            <script>
            document.addEventListener('DOMContentLoaded', function() {
                var recipientInput = document.getElementById('recipient_email');
                var creditCounter = document.getElementById('credit-counter');
                var fileInput = document.querySelector('input[type="file"]');
                var fileCounter = document.getElementById('file-counter');
                var subjectInput = document.querySelector('input[name="subject"]');
                var form = document.getElementById('legallymail-send-form');

                function updateCredits() {
                    var val = recipientInput.value;
                    if (!val.trim()) {
                        creditCounter.textContent = '0';
                        return;
                    }
                    var recipients = val.split(',').filter(function(email) {
                        return email.trim().length > 0;
                    });
                    creditCounter.textContent = recipients.length;
                }

                if (recipientInput) {
                    recipientInput.addEventListener('input', updateCredits);
                    updateCredits();
                }

                if (fileInput) {
                    fileInput.addEventListener('change', function() {
                        var files = this.files;
                        var totalSize = 0;
                        var maxBytes = 15 * 1024 * 1024; // 15 MB
                        var allowedExtensions = ['pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'];

                        for (var i = 0; i < files.length; i++) {
                            var ext = files[i].name.split('.').pop().toLowerCase();
                            if (!allowedExtensions.includes(ext)) {
                                 alert('File type not allowed: ' + files[i].name + '. Only PDF, DOC, DOCX, JPG, PNG are permitted.');
                                 this.value = '';
                                 fileCounter.textContent = '0 MB';
                                 return;
                            }
                            totalSize += files[i].size;
                        }
                        
                        var sizeMB = (totalSize / (1024*1024)).toFixed(2);
                        fileCounter.textContent = sizeMB + ' MB';

                        if (totalSize > maxBytes) {
                            alert('Warning: Total file size (' + sizeMB + ' MB) exceeds the 15 MB limit. Please remove some files.');
                            this.value = '';
                            fileCounter.textContent = '0 MB';
                        }
                    });
                }

                if (form) {
                    form.addEventListener('submit', function(e) {
                        var subject = subjectInput.value.trim();
                        
                        // Subject Validation (min 2 chars)
                        if (subject.length < 2) {
                            alert('Subject must be at least 2 characters long.');
                            e.preventDefault();
                            return;
                        }

                        // Message Validation (min 10 chars)
                        var messageContent = '';
                        if (typeof tinyMCE !== 'undefined' && tinyMCE.get('message')) {
                            messageContent = tinyMCE.get('message').getContent({format: 'text'}).trim();
                        } else {
                            var msgField = document.getElementById('message');
                            if (msgField) {
                                messageContent = msgField.value.trim();
                            }
                        }

                        if (messageContent.length < 10) {
                            alert('Message must be at least 10 characters long.');
                            e.preventDefault();
                            return;
                        }

                        // Recipient Validation (check if enough credits)
                        var available = '<?php echo esc_js($available_credits); ?>';
                        var needed = parseInt(creditCounter.textContent);
                        
                        if (!isNaN(available)) {
                            var availableInt = parseInt(available);
                            if (needed > availableInt) {
                                alert('Error: You do not have enough credits. Required: ' + needed + ', Available: ' + availableInt);
                                e.preventDefault();
                                return;
                            }
                        }
                    });
                }

                // Template Logic
                var templateSelector = document.getElementById('template-selector');
                var btnLoad = document.getElementById('btn-load-template');
                var btnSave = document.getElementById('btn-save-template');
                var btnDelete = document.getElementById('btn-delete-template');

                btnLoad.addEventListener('click', function() {
                    var option = templateSelector.options[templateSelector.selectedIndex];
                    if (!option.value) return;

                    var subject = option.getAttribute('data-subject');
                    var message = option.getAttribute('data-message');

                    if (subject) subjectInput.value = subject;
                    
                    if (typeof tinyMCE !== 'undefined' && tinyMCE.get('message')) {
                        tinyMCE.get('message').setContent(message);
                    } else if (document.getElementById('message')) {
                        document.getElementById('message').value = message;
                    }
                });

                btnSave.addEventListener('click', function() {
                    var name = prompt("Enter a name for this template:");
                    if (!name) return;

                    var subject = subjectInput.value;
                    var message = '';
                    if (typeof tinyMCE !== 'undefined' && tinyMCE.get('message')) {
                        message = tinyMCE.get('message').getContent();
                    } else {
                        message = document.getElementById('message').value;
                    }

                    jQuery.post(ajaxurl, {
                        action: 'legallymail_save_template',
                        nonce: '<?php echo esc_js( wp_create_nonce( 'legallymail_template_action' ) ); ?>',
                        name: name,
                        subject: subject,
                        message: message
                    }, function(res) {
                        if (res.success) {
                            alert('Template saved!');
                            location.reload(); 
                        } else {
                            alert('Error saving template: ' + res.data);
                        }
                    });
                });

                btnDelete.addEventListener('click', function() {
                    var id = templateSelector.value;
                    if (!id) return;
                    if (!confirm('Are you sure you want to delete this template?')) return;

                    jQuery.post(ajaxurl, {
                        action: 'legallymail_delete_template',
                        nonce: '<?php echo esc_js( wp_create_nonce( 'legallymail_template_action' ) ); ?>',
                        id: id
                    }, function(res) {
                        if (res.success) {
                            alert('Template deleted!');
                            location.reload();
                        }
                    });
                });
            });
            </script>
        </div>
        <?php
    }

    public function display_email_list_page() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'legallymail_emails';
        
        // CSS for Badges
        echo '<style>
            .status-badge { padding: 5px 10px; border-radius: 4px; font-weight: bold; color: #fff; font-size: 12px; }
            .status-queued, .status-pending { background-color: #999; }
            .status-processing { background-color: #0073aa; } /* Blue */
            .status-sent { background-color: #0073aa; } /* Blue */
            .status-delivered { background-color: #46b450; } /* Green */
            .status-failed { background-color: #dc3232; } /* Red */
            .status-read { background-color: #6f42c1; } /* Purple */
            .filter-form { margin-bottom: 20px; background: #fff; padding: 15px; border: 1px solid #ccd0d4; box-shadow: 0 1px 1px rgba(0,0,0,.04); }
            .filter-form input, .filter-form select { margin-right: 10px; }
        </style>';

        // Check for success message
        $message_param = isset( $_GET['message'] ) ? sanitize_text_field( wp_unslash( $_GET['message'] ) ) : '';
        if ( 'sent' === $message_param ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Certified Email sent successfully!', 'legallymail-certified-email' ) . '</p></div>';
        }

        // Handle Status Check Action
        $action_param = isset( $_GET['action'] ) ? sanitize_text_field( wp_unslash( $_GET['action'] ) ) : '';
        $id_param     = isset( $_GET['id'] ) ? intval( wp_unslash( $_GET['id'] ) ) : 0;

        if ( 'check_status' === $action_param && $id_param > 0 ) {
            $id = $id_param;
            $email_record = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM `{$wpdb->prefix}legallymail_emails` WHERE id = %d", $id ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            
            if ( $email_record && !empty( $email_record->tracking_id ) ) {
                $status_data = $this->api->get_email_status( $email_record->tracking_id );
                if ( !is_wp_error( $status_data ) ) {
                    $api_status = isset($status_data['status']) ? $status_data['status'] : $email_record->status;
                    $api_recipients = isset($status_data['recipients']) ? $status_data['recipients'] : [];
                    
                    if (!empty($api_recipients)) {
                        foreach ($api_recipients as $recipient) {
                            $wpdb->update(
                                $table_name,
                                array(
                                    'status' => $recipient['status'],
                                    'error_status' => isset($recipient['error_details']) ? $recipient['error_details'] : null
                                ),
                                array('tracking_id' => $email_record->tracking_id, 'recipient' => $recipient['email'])
                            );
                        }
                        $new_status = $wpdb->get_var($wpdb->prepare("SELECT status FROM `{$wpdb->prefix}legallymail_emails` WHERE id = %d", $id)); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                    } else {
                        $new_status = $api_status;
                        $error_status = isset($status_data['error_status']) ? $status_data['error_status'] : null;
                        $wpdb->update( 
                            $table_name, 
                            array( 
                                'status' => $new_status,
                                'error_status' => $error_status
                            ), 
                            array( 'id' => $id ) 
                        );
                    }
                    
                    if (!$new_status) $new_status = $api_status;
                    echo '<div class="notice notice-success is-dismissible"><p>Status updated to: ' . esc_html( ucfirst( $new_status ) ) . '</p></div>';
                }
            }
        }

        // Filters
        $filter_email  = isset( $_GET['filter_email'] ) ? sanitize_text_field( wp_unslash( $_GET['filter_email'] ) ) : '';
        $filter_status = isset( $_GET['filter_status'] ) ? sanitize_text_field( wp_unslash( $_GET['filter_status'] ) ) : '';
        $paged_param   = isset( $_GET['paged'] ) ? max( 1, intval( wp_unslash( $_GET['paged'] ) ) ) : 1;

        $where_clauses = array( '1=1' );
        $query_args    = array();

        if ( ! empty( $filter_email ) ) {
            $where_clauses[] = 'recipient LIKE %s';
            $query_args[]    = '%' . $wpdb->esc_like( $filter_email ) . '%';
        }

        if ( ! empty( $filter_status ) ) {
            $where_clauses[] = 'status = %s';
            $query_args[]    = $filter_status;
        }

        $where_sql = implode( ' AND ', $where_clauses );
        
        // Pagination logic
        $per_page     = 20;
        $current_page = $paged_param;
        $offset       = ( $current_page - 1 ) * $per_page;

        $total_items = 0;
        $table_raw   = $wpdb->prefix . 'legallymail_emails';

        if ( ! empty( $query_args ) ) {
            $total_items = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM `$table_raw` WHERE $where_sql", $query_args ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $results     = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `$table_raw` WHERE $where_sql ORDER BY time DESC LIMIT %d OFFSET %d", array_merge( $query_args, array( $per_page, $offset ) ) ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        } else {
            $total_items = (int) $wpdb->get_var( "SELECT COUNT(*) FROM `$table_raw` WHERE $where_sql" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
            $results     = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM `$table_raw` WHERE $where_sql ORDER BY time DESC LIMIT %d OFFSET %d", array( $per_page, $offset ) ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter
        }
        $total_pages = ceil($total_items / $per_page);
        ?>
        <div class="wrap">
            <h1>Create and Manage Certified Emails</h1>
            <a href="admin.php?page=legallymail-send-email" class="page-title-action">Send New</a>
            
            <div class="filter-form">
                <form method="get" action="">
                    <input type="hidden" name="page" value="legallymail-certified-email" />
                    <label>Recipient: <input type="text" name="filter_email" value="<?php echo esc_attr($filter_email); ?>" placeholder="Search by email..." /></label>
                    <label>Status: 
                        <select name="filter_status">
                            <option value="">All Statuses</option>
                            <option value="sent" <?php selected($filter_status, 'sent'); ?>>Sent</option>
                            <option value="delivered" <?php selected($filter_status, 'delivered'); ?>>Delivered</option>
                            <option value="read" <?php selected($filter_status, 'read'); ?>>Read</option>
                            <option value="failed" <?php selected($filter_status, 'failed'); ?>>Failed</option>
                        </select>
                    </label>
                    <input type="submit" class="button" value="Filter" />
                    <?php if(!empty($filter_email) || !empty($filter_status)): ?>
                        <a href="admin.php?page=legallymail-certified-email" class="button">Reset</a>
                    <?php endif; ?>
                    <button type="button" id="bulk-check-status" class="button button-secondary" style="margin-left: 10px;">Update All Statuses</button>
                    <?php wp_nonce_field('legallymail_bulk_update', 'legallymail_bulk_nonce'); ?>
                </form>
            </div>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Recipient</th>
                        <th>Subject</th>
                        <th>Status</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( $results ) : ?>
                        <?php foreach ( $results as $row ) : ?>
                            <tr>
                                <td><?php echo esc_html( gmdate( 'Y-m-d H:i', strtotime( $row->time ) ) ); ?></td>
                                <td><?php echo wp_kses( str_replace( ',', '<br>', esc_html( $row->recipient ) ), array( 'br' => array() ) ); ?></td>
                                <td><?php echo esc_html( $row->subject ); ?></td>
                                <td>
                                    <?php 
                                    $status = $row->status;
                                    $status_class = 'status-queued'; // default
                                    
                                    // Map statuses to meaningful classes
                                    switch($status) {
                                        case 'processed':
                                        case 'sent':
                                            $status_class = 'status-sent';
                                            break;
                                        case 'delivered':
                                            $status_class = 'status-delivered';
                                            break;
                                        case 'read':
                                        case 'opened':
                                            $status_class = 'status-read';
                                            break;
                                        case 'failed':
                                        case 'bounced':
                                            $status_class = 'status-failed';
                                            break;
                                        case 'pending':
                                            $status_class = 'status-pending';
                                            break;
                                        default:
                                            $status_class = 'status-queued';
                                    }
                                    
                                    echo '<span class="status-badge ' . esc_attr( $status_class ) . '" data-id="' . esc_attr( $row->id ) . '" data-status="' . esc_attr( $status ) . '">' . esc_html( ucfirst( $status ) ) . '</span>'; 

                                    // Show error info if failed
                                    if ( $status == 'failed' || $status == 'bounced' ) {
                                        $error_details = 'Unknown error';
                                        if ( !empty($row->error_status) ) {
                                            $error_details = $row->error_status;
                                        } elseif ( !empty($row->api_response) ) {
                                            $response_data = json_decode($row->api_response, true);
                                            // Handle both structured error and raw API response format in api_response
                                            if ( isset($response_data['error']) ) {
                                                $error_details = is_array($response_data['error']) ? json_encode($response_data['error']) : $response_data['error'];
                                            } elseif ( isset($response_data['message']) ) {
                                                $error_details = $response_data['message'];
                                            } elseif ( is_string($response_data) ) {
                                                $error_details = $response_data;
                                            }
                                        }
                                        echo ' <span class="dashicons dashicons-info error-info-icon" title="' . esc_attr($error_details) . '" data-error="' . esc_attr($error_details) . '" style="color: #dc3232; cursor: pointer; vertical-align: middle;"></span>';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <a href="<?php echo esc_url( admin_url( 'admin.php?page=legallymail-certified-email&action=check_status&id=' . $row->id ) ); ?>" class="button button-small">Check Status</a>
                                    <?php if( !empty($row->tracking_id) ): ?>
                                    <a href="https://legallymail.com/emails/view?id=<?php echo esc_attr( $row->tracking_id ); ?>" target="_blank" class="button button-small">Details</a>
                                    <a href="https://legallymail.com/emails/download-certificate?id=<?php echo esc_attr( $row->tracking_id ); ?>&realtime=1" target="_blank" class="button button-small button-primary">Download Certificate</a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr><td colspan="5">No certified emails sent yet.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <?php if ($total_pages > 1): ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <span class="displaying-num"><?php echo esc_html( $total_items ); ?> items</span>
                        <span class="pagination-links">
                            <?php if ($current_page > 1): ?>
                                <a class="prev-page button" href="<?php echo esc_url( add_query_arg( 'paged', $current_page - 1 ) ); ?>"><span class="screen-reader-text">Previous page</span><span aria-hidden="true">‹</span></a>
                            <?php endif; ?>
                            <span class="paging-input">
                                <span class="tablenav-paging-text"><?php echo esc_html( $current_page ); ?> of <span class="total-pages"><?php echo esc_html( $total_pages ); ?></span></span>
                            </span>
                            <?php if ($current_page < $total_pages): ?>
                                <a class="next-page button" href="<?php echo esc_url( add_query_arg( 'paged', $current_page + 1 ) ); ?>"><span class="screen-reader-text">Next page</span><span aria-hidden="true">›</span></a>
                            <?php endif; ?>
                        </span>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Mobile friendly error display
            var errorIcons = document.querySelectorAll('.error-info-icon');
            errorIcons.forEach(function(icon) {
                icon.addEventListener('click', function(e) {
                    var errorMsg = this.getAttribute('data-error');
                    if (errorMsg) {
                        alert('Error Details:\n' + errorMsg);
                    }
                });
            });

            var bulkBtn = document.getElementById('bulk-check-status');
            if (bulkBtn) {
                bulkBtn.addEventListener('click', function() {
                    var badges = document.querySelectorAll('.status-badge');
                    var idsToUpdate = [];
                    
                    badges.forEach(function(badge) {
                        var status = badge.getAttribute('data-status');
                        var id = badge.getAttribute('data-id');
                        if (status !== 'read' && status !== 'failed' && status !== 'bounced' && status !== 'opened') {
                            idsToUpdate.push(id);
                        }
                    });

                    if (idsToUpdate.length === 0) {
                        alert('No emails need updating.');
                        return;
                    }

                    if (!confirm('Update status for ' + idsToUpdate.length + ' emails?')) {
                        return;
                    }

                    // Rate limiting: 2 times per minute
                    var now = Date.now();
                    var clicks = JSON.parse(localStorage.getItem('legallymail_bulk_clicks') || '[]');
                    
                    // Filter clicks in the last 60 seconds
                    clicks = clicks.filter(function(timestamp) {
                        return now - timestamp < 60000;
                    });

                    if (clicks.length >= 2) {
                        var waitTime = Math.ceil((60000 - (now - clicks[0])) / 1000);
                        alert('Rate limit reached. Please wait ' + waitTime + ' seconds before updating again.');
                        return;
                    }

                    // Log this click
                    clicks.push(now);
                    localStorage.setItem('legallymail_bulk_clicks', JSON.stringify(clicks));

                    bulkBtn.disabled = true;
                    bulkBtn.textContent = 'Updating...';

                    var nonce = document.getElementById('legallymail_bulk_nonce').value;
                    var processed = 0;

                    function updateNext() {
                        if (processed >= idsToUpdate.length) {
                            bulkBtn.disabled = false;
                            bulkBtn.textContent = 'Update All Statuses';
                            alert('Update complete!');
                            return;
                        }

                        var currentId = idsToUpdate[processed];
                        var badge = document.querySelector('.status-badge[data-id="' + currentId + '"]');
                        if (badge) {
                            badge.style.opacity = '0.5';
                        }

                        jQuery.post(ajaxurl, {
                            action: 'legallymail_update_status',
                            id: currentId,
                            nonce: nonce
                        }, function(response) {
                            if (response.success) {
                                if (badge) {
                                    badge.textContent = response.data.status_label;
                                    badge.setAttribute('data-status', response.data.status);
                                    
                                    // Update classes
                                    badge.className = 'status-badge status-' + response.data.status;
                                    badge.style.opacity = '1';

                                    // Handle error icon
                                    var parentTd = badge.parentElement;
                                    var existingIcon = parentTd.querySelector('.error-info-icon');
                                    
                                    if (response.data.status === 'failed' || response.data.status === 'bounced') {
                                        var errorMsg = response.data.error_status || 'Unknown error';
                                        if (existingIcon) {
                                            existingIcon.setAttribute('title', errorMsg);
                                            existingIcon.setAttribute('data-error', errorMsg);
                                        } else {
                                            var newIcon = document.createElement('span');
                                            newIcon.className = 'dashicons dashicons-info error-info-icon';
                                            newIcon.setAttribute('title', errorMsg);
                                            newIcon.setAttribute('data-error', errorMsg);
                                            newIcon.style.color = '#dc3232';
                                            newIcon.style.cursor = 'pointer';
                                            newIcon.style.verticalAlign = 'middle';
                                            newIcon.style.marginLeft = '5px';
                                            newIcon.onclick = function() { alert('Error Details:\n' + errorMsg); };
                                            parentTd.appendChild(newIcon);
                                        }
                                    } else if (existingIcon) {
                                        existingIcon.remove();
                                    }
                                }
                            }
                            processed++;
                            updateNext();
                        });
                    }

                    updateNext();
                });
            }
        });
        </script>
        <?php
    }

    public function ajax_update_status() {
        check_ajax_referer('legallymail_bulk_update', 'nonce');
        
        $id = isset($_POST['id']) ? intval( wp_unslash( $_POST['id'] ) ) : 0;
        if (!$id) {
            wp_send_json_error('Invalid ID');
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'legallymail_emails';
        $email_record = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $id));

        if ($email_record && !empty( $email_record->tracking_id )) {
            $status_data = $this->api->get_email_status($email_record->tracking_id);
            if (!is_wp_error($status_data)) {
                $api_status = isset($status_data['status']) ? $status_data['status'] : $email_record->status;
                $api_recipients = isset($status_data['recipients']) ? $status_data['recipients'] : [];
                
                if (!empty($api_recipients)) {
                    // Update all rows for this tracking_id if it's a multi-recipient send
                    foreach ($api_recipients as $recipient) {
                        $wpdb->update(
                            $table_name,
                            array(
                                'status' => $recipient['status'],
                                'error_status' => isset($recipient['error_details']) ? $recipient['error_details'] : null
                            ),
                            array('tracking_id' => $email_record->tracking_id, 'recipient' => $recipient['email'])
                        );
                    }
                    // Fetch the updated status for the specific row requested
                    $row = $wpdb->get_row($wpdb->prepare("SELECT status, error_status FROM $table_name WHERE id = %d", $id));
                    $new_status = $row->status;
                    $error_status = $row->error_status;
                } else {
                    $new_status = $api_status;
                    $error_status = isset($status_data['error_status']) ? $status_data['error_status'] : null;
                    $wpdb->update(
                        $table_name,
                        array(
                            'status' => $new_status,
                            'error_status' => $error_status
                        ),
                        array('id' => $id)
                    );
                }

                if (!$new_status) $new_status = $api_status;
                
                // Normalizar estados para las clases CSS
                $normalized_status = $new_status;
                if ($new_status == 'processed') $normalized_status = 'sent';
                if ($new_status == 'opened') $normalized_status = 'read';
                if ($new_status == 'bounced') $normalized_status = 'failed';

                wp_send_json_success(array(
                    'status' => $normalized_status, 
                    'status_label' => ucfirst($new_status),
                    'error_status' => $error_status
                ));
            }
        }
        wp_send_json_error('API Error or record not found');
    }
}
