import { LightningElement, api, track, wire } from 'lwc';
import { getRecord, getFieldValue } from 'lightning/uiRecordApi';
import { ShowToastEvent } from 'lightning/platformShowToastEvent';
import { refreshApex } from '@salesforce/apex';
import sendEmail from '@salesforce/apex/LegallyMailService.sendCertifiedEmail';
import getTemplates from '@salesforce/apex/LegallyMailService.getTemplates';
import saveTemplate from '@salesforce/apex/LegallyMailService.saveTemplate';

// Campos dinámicos para detectar el email
import CONTACT_EMAIL from '@salesforce/schema/Contact.Email';
import LEAD_EMAIL from '@salesforce/schema/Lead.Email';

export default class LegallyMailSendForm extends LightningElement {
    @api recordId; // ID del registro actual (Contacto o Candidate)
    @api objectApiName;

    @track recipientEmail = '';
    @track subject = '';
    @track message = '';
    @track files = [];
    @track isLoading = false;

    @track templateOptions = [];
    selectedTemplateId = '';
    wiredTemplatesResult;

    acceptedFormats = ['.pdf', '.doc', '.docx', '.jpg', '.jpeg', '.png'];

    // Obtener plantillas
    @wire(getTemplates)
    wiredTemplates(result) {
        this.wiredTemplatesResult = result;
        if (result.data) {
            this.templateOptions = result.data.map(t => {
                return { label: t.Name, value: t.Id };
            });
        }
    }

    // Auto-detectar email según el objeto
    @wire(getRecord, { recordId: '$recordId', fields: [CONTACT_EMAIL, LEAD_EMAIL] })
    wiredRecord({ error, data }) {
        if (data) {
            const email = getFieldValue(data, CONTACT_EMAIL) || getFieldValue(data, LEAD_EMAIL);
            if (email) this.recipientEmail = email;
        }
    }

    get fileNames() {
        return this.files.map(f => f.name).join(', ');
    }

    handleEmailChange(event) { this.recipientEmail = event.target.value; }
    handleSubjectChange(event) { this.subject = event.target.value; }
    handleMessageChange(event) { this.message = event.target.value; }

    handleTemplateChange(event) {
        this.selectedTemplateId = event.detail.value;
        const selectedTmpl = this.wiredTemplatesResult.data.find(t => t.Id === this.selectedTemplateId);
        if (selectedTmpl) {
            this.subject = selectedTmpl.Subject__c;
            this.message = selectedTmpl.Body__c;
        }
    }

    async handleSaveTemplate() {
        const name = prompt('Enter a name for this template:');
        if (!name) return;

        if (!this.subject || !this.message) {
            this.showToast('Error', 'Subject and Message are required to save a template', 'error');
            return;
        }

        try {
            await saveTemplate({ name, subject: this.subject, body: this.message });
            this.showToast('Success', 'Template saved successfully', 'success');
            await refreshApex(this.wiredTemplatesResult);
        } catch (error) {
            this.showToast('Error', 'Failed to save template', 'error');
        }
    }

    handleUploadFinished(event) {
        const uploadedFiles = event.detail.files;
        if (this.files.length + uploadedFiles.length > 4) {
            this.showToast('Error', 'Maximum 4 attachments allowed', 'error');
            return;
        }
        this.files = [...this.files, ...uploadedFiles];
    }

    async handleSend() {
        if (!this.recipientEmail || !this.subject || !this.message) {
            this.showToast('Error', 'Please fill all required fields', 'error');
            return;
        }

        this.isLoading = true;
        try {
            // Nota: En Salesforce los archivos se pasan como Base64 o IDs de ContentVersion
            // Aquí llamaríamos al Apex pasando los datos necesarios.
            const result = await sendEmail({
                recipient: this.recipientEmail,
                subject: this.subject,
                message: this.message,
                attachments: [] // En una implementación completa convertiríamos los IDs a Mapas de contenido
            });

            if (result.success) {
                this.showToast('Success', 'Certified Email sent! Tracking ID: ' + result.data.tracking_id, 'success');
                this.resetForm();
            }
        } catch (error) {
            this.showToast('API Error', error.body.message, 'error');
        } finally {
            this.isLoading = false;
        }
    }

    showToast(title, message, variant) {
        this.dispatchEvent(new ShowToastEvent({ title, message, variant }));
    }

    resetForm() {
        this.subject = '';
        this.message = '';
        this.files = [];
    }
}
