import { LightningElement, api, track, wire } from 'lwc';
import { ShowToastEvent } from 'lightning/platformShowToastEvent';
import getEmailsFromIds from '@salesforce/apex/LegallyMailService.getEmailsFromIds';
import getTemplates from '@salesforce/apex/LegallyMailService.getTemplates';
import sendCertifiedEmail from '@salesforce/apex/LegallyMailService.sendCertifiedEmail';

export default class LegallyMailBulkSend extends LightningElement {
    @api ids; // Salesforce passes selected record IDs here for Mass Actions

    @track subject = '';
    @track message = '';
    @track recipients = [];
    @track isLoadingInitial = true;
    @track isSending = false;

    @track templateOptions = [];
    selectedTemplateId = '';
    wiredTemplatesResult;

    @wire(getTemplates)
    wiredTemplates(result) {
        this.wiredTemplatesResult = result;
        if (result.data) {
            this.templateOptions = result.data.map(t => ({ label: t.Name, value: t.Id }));
        }
    }

    connectedCallback() {
        if (this.ids && this.ids.length > 0) {
            this.loadRecipients();
        } else {
            this.isLoadingInitial = false;
        }
    }

    async loadRecipients() {
        try {
            const emailMap = await getEmailsFromIds({ recordIds: this.ids });
            this.recipients = Object.values(emailMap);
            this.isLoadingInitial = false;
        } catch (error) {
            this.showToast('Error', 'Failed to load recipients', 'error');
            this.isLoadingInitial = false;
        }
    }

    get recipientCount() {
        return this.recipients.length;
    }

    get isSendDisabled() {
        return this.isSending || !this.subject || !this.message || this.recipientCount === 0;
    }

    handleTemplateChange(event) {
        this.selectedTemplateId = event.detail.value;
        const selectedTmpl = this.wiredTemplatesResult.data.find(t => t.Id === this.selectedTemplateId);
        if (selectedTmpl) {
            this.subject = selectedTmpl.Subject__c;
            this.message = selectedTmpl.Body__c;
        }
    }

    handleSubjectChange(event) {
        this.subject = event.target.value;
    }

    handleMessageChange(event) {
        this.message = event.target.value;
    }

    async handleSendBulk() {
        this.isSending = true;
        let successCount = 0;
        let errorCount = 0;

        for (const email of this.recipients) {
            try {
                const res = await sendCertifiedEmail({
                    recipient: email,
                    subject: this.subject,
                    message: this.message,
                    attachments: []
                });
                if (res.success) successCount++;
                else errorCount++;
            } catch (err) {
                errorCount++;
            }
        }

        this.showToast('Bulk Send Finished', `Successfully sent: ${successCount}, Errors: ${errorCount}`, successCount > 0 ? 'success' : 'error');
        this.handleCancel();
    }

    handleCancel() {
        // Cierra la acción rápida
        this.dispatchEvent(new CustomEvent('close'));
    }

    showToast(title, message, variant) {
        this.dispatchEvent(new ShowToastEvent({ title, message, variant }));
    }
}
