public with sharing class LegallyMailService {
    
    // Obtenemos la configuración desde Custom Metadata o Custom Settings
    // Para simplificar, asumimos que existe un Named Credential o se configura vía Apex
    private static String getApiToken() {
        // En producción se usaría: return [SELECT Token__c FROM LegallyMail_Setting__mdt LIMIT 1].Token__c;
        return 'YOUR_API_TOKEN'; 
    }

    private static String getApiUrl() {
        return 'https://legallymail.com';
    }

    @AuraEnabled(cacheable=true)
    public static Map<String, String> getEmailsFromIds(List<Id> recordIds) {
        Map<String, String> idToEmail = new Map<String, String>();
        if (recordIds == null || recordIds.isEmpty()) return idToEmail;

        String sObjectType = recordIds[0].getSObjectType().getDescribe().getName();
        
        if (sObjectType == 'Contact') {
            for (Contact c : [SELECT Id, Email FROM Contact WHERE Id IN :recordIds AND Email != null]) {
                idToEmail.put(c.Id, c.Email);
            }
        } else if (sObjectType == 'Lead') {
            for (Lead l : [SELECT Id, Email FROM Lead WHERE Id IN :recordIds AND Email != null]) {
                idToEmail.put(l.Id, l.Email);
            }
        }
        return idToEmail;
    }

    @AuraEnabled
    public static Map<String, Object> sendCertifiedEmail(String recipient, String subject, String message, List<Map<String, String>> attachments) {
        String token = getApiToken();
        String url = getApiUrl() + '/api/send';

        Map<String, Object> payload = new Map<String, Object>();
        payload.put('token', token);
        payload.put('recipient', recipient);
        payload.put('subject', subject);
        payload.put('message', message);
        payload.put('attachments', attachments);

        Http http = new Http();
        HttpRequest request = new HttpRequest();
        request.setEndpoint(url);
        request.setMethod('POST');
        request.setHeader('Content-Type', 'application/json');
        request.setHeader('Authorization', 'Bearer ' + token);
        request.setBody(JSON.serialize(payload));
        request.setTimeout(30000);

        try {
            HttpResponse response = http.send(request);
            Map<String, Object> result = (Map<String, Object>) JSON.deserializeUntyped(response.getBody());
            
            if (response.getStatusCode() == 200 && (Boolean)result.get('success')) {
                // Guardar el Log en Salesforce
                Map<String, Object> data = (Map<String, Object>)result.get('data');
                String trackingId = (String) (data != null ? data.get('tracking_id') : result.get('tracking_id'));
                
                LegallyMail_Log__c log = new LegallyMail_Log__c();
                log.Recipient__c = recipient;
                log.Subject__c = subject;
                log.Tracking_ID__c = trackingId;
                log.Status__c = 'sent';
                insert log;

                return result;
            } else {
                throw new AuraHandledException('API Error: ' + result.get('error'));
            }
        } catch (Exception e) {
            throw new AuraHandledException('Connection Error: ' + e.getMessage());
        }
    }

    @AuraEnabled(cacheable=true)
    public static List<LegallyMail_Template__c> getTemplates() {
        return [SELECT Id, Name, Subject__c, Body__c FROM LegallyMail_Template__c ORDER BY Name ASC];
    }

    @AuraEnabled
    public static LegallyMail_Template__c saveTemplate(String name, String subject, String body) {
        LegallyMail_Template__c tmpl = new LegallyMail_Template__c(
            Name = name,
            Subject__c = subject,
            Body__c = body
        );
        insert tmpl;
        return tmpl;
    }

    @AuraEnabled
    public static Map<String, Object> getStatus(String trackingId) {
        String token = getApiToken();
        String url = getApiUrl() + '/api/status/' + trackingId;

        Http http = new Http();
        HttpRequest request = new HttpRequest();
        request.setEndpoint(url);
        request.setMethod('GET');
        request.setHeader('Authorization', 'Bearer ' + token);

        try {
            HttpResponse response = http.send(request);
            return (Map<String, Object>) JSON.deserializeUntyped(response.getBody());
        } catch (Exception e) {
            throw new AuraHandledException('Status Check Failed: ' + e.getMessage());
        }
    }
}
