from odoo import api, fields, models, _
from odoo.exceptions import UserError
import requests
import base64
import json
import logging

_logger = logging.getLogger(__name__)

class SendCertifiedEmailWizard(models.TransientModel):
    _name = 'legallymail.send.wizard'
    _description = 'Send Certified Email Wizard'

    recipient_emails = fields.Text(string='Recipient Emails', required=True, help="Comma separated list of email addresses")
    subject = fields.Char(string='Subject', required=True)
    body_html = fields.Html(string='Message', required=True)
    template_id = fields.Many2one('legallymail.template', string='Use Template')
    attachment_ids = fields.Many2many('ir.attachment', string='Attachments')

    @api.onchange('template_id')
    def _onchange_template_id(self):
        if self.template_id:
            self.subject = self.template_id.subject
            self.body_html = self.template_id.body_html

    def action_send_email(self):
        self.ensure_one()
        api_token = self.env['ir.config_parameter'].sudo().get_param('legallymail.api_token')
        api_url = self.env['ir.config_parameter'].sudo().get_param('legallymail.api_url') or 'https://legallymail.com'

        if not api_token:
            raise UserError(_("LegallyMail API Token is not configured."))

        recipients = [email.strip() for email in self.recipient_emails.split(',') if email.strip()]
        
        if not recipients:
            raise UserError(_("Please specify at least one recipient."))

        attachments_data = []
        for attachment in self.attachment_ids:
            attachments_data.append({
                'name': attachment.name,
                'content': attachment.datas.decode('utf-8') if isinstance(attachment.datas, bytes) else attachment.datas, # Ensure string
                'type': attachment.mimetype,
            })

        # The API seems to handle multiple recipients if sent as comma-separated string based on WP plugin logic?
        # WP plugin: $recipient_list = explode(',', $to); ... API call takes $to directly.
        # So yes, API likely handles comma-separated string or list.
        # WP: $body['recipient'] = $to;
        # So I will pass the raw string.

        payload = {
            'token': api_token,
            'recipient': self.recipient_emails,
            'subject': self.subject,
            'message': self.body_html, # HTML content
            'attachments': attachments_data
        }

        headers = {
            'Authorization': f'Bearer {api_token}',
            'Content-Type': 'application/json',
        }

        try:
            response = requests.post(f"{api_url}/api/send", json=payload, headers=headers, timeout=30)
            response.raise_for_status()
            result = response.json()

            if result.get('success'):
                tracking_id = result.get('data', {}).get('tracking_id') or result.get('tracking_id')
                
                # Create log entries
                for email in recipients:
                    self.env['legallymail.log'].create({
                        'recipient': email,
                        'subject': self.subject,
                        'tracking_id': tracking_id,
                        'status': 'sent', # Assume sent initially
                        'api_response': json.dumps(result),
                    })
                
                return {
                    'type': 'ir.actions.client',
                    'tag': 'display_notification',
                    'params': {
                        'title': _('Success'),
                        'message': _('Certified email sent successfully!'),
                        'type': 'success',
                        'sticky': False,
                        'next': {'type': 'ir.actions.act_window_close'},
                    }
                }
            else:
                error = result.get('error', 'Unknown Error')
                raise UserError(_("API Error: %s") % error)

        except Exception as e:
            raise UserError(_("Failed to send email: %s") % str(e))
