from odoo import api, fields, models, _
import requests
import json
import logging

_logger = logging.getLogger(__name__)

class LegallyMailLog(models.Model):
    _name = 'legallymail.log'
    _description = 'LegallyMail Email Log'
    _order = 'create_date desc'

    recipient = fields.Char(string='Recipient', required=True)
    subject = fields.Char(string='Subject', required=True)
    tracking_id = fields.Char(string='Tracking ID', readonly=True)
    status = fields.Selection([
        ('queued', 'Queued'),
        ('processed', 'Processed'),
        ('sent', 'Sent'),
        ('delivered', 'Delivered'),
        ('read', 'Read'),
        ('opened', 'Opened'),
        ('clicked', 'Clicked'),
        ('failed', 'Failed'),
        ('bounced', 'Bounced'),
    ], string='Status', default='queued')
    api_response = fields.Text(string='API Response', readonly=True)
    error_status = fields.Text(string='Error Details', readonly=True)

    def name_get(self):
        result = []
        for record in self:
            result.append((record.id, f"{record.subject} ({record.recipient})"))
        return result

    def check_status(self):
        self.ensure_one()
        if not self.tracking_id:
            return

        api_token = self.env['ir.config_parameter'].sudo().get_param('legallymail.api_token')
        api_url = self.env['ir.config_parameter'].sudo().get_param('legallymail.api_url')
        
        if not api_token:
            _logger.warning("LegallyMail API Token not configured")
            return

        url = f"{api_url}/api/email"
        headers = {
            'Authorization': f'Bearer {api_token}',
            'Content-Type': 'application/json',
        }
        params = {'tracking_id': self.tracking_id}

        try:
            response = requests.get(url, headers=headers, params=params, timeout=10)
            response.raise_for_status()
            data = response.json()

            if data.get('success'):
                email_data = data.get('email', {})
                # The API returns 'recipients' list for multi-recipient, but we store one per row usually?
                # The WP plugin handles multi-recipient by creating multiple rows.
                # Assuming this tracking_id is unique per send, but might cover multiple recipients if sent in batch.
                # If we store one row per recipient, we filter by recipient email.
                
                new_status = email_data.get('status')
                # If there are recipients detail, try to find ours
                if 'recipients' in email_data:
                    for r in email_data['recipients']:
                        if r['email'] == self.recipient:
                            new_status = r['status']
                            if 'error_details' in r:
                                self.error_status = r['error_details']
                            break
                            
                if new_status:
                    # Check if status is valid option
                    status_key = new_status.lower()
                    # Check if status_key is in selection keys
                    selection_keys = [s[0] for s in self._fields['status'].selection]
                    if status_key in selection_keys:
                        self.status = status_key
                    else:
                        # Fallback or log warning? For now just log
                        _logger.warning(f"Unknown status {new_status} for tracking_id {self.tracking_id}")
                
                # Check for top level error if failed
                if new_status == 'failed' and not self.error_status:
                    self.error_status = email_data.get('error_status')

        except Exception as e:
            _logger.error(f"Failed to check status for {self.tracking_id}: {e}")

    def action_check_status(self):
        for record in self:
            record.check_status()

    def action_open_certificate(self):
        self.ensure_one()
        if self.tracking_id:
            url = f"https://legallymail.com/emails/download-certificate?id={self.tracking_id}&realtime=1"
            return {
                'type': 'ir.actions.act_url',
                'url': url,
                'target': 'new',
            }
